# encoding=utf-8

__author__ = "Roland Trouville"
__copyright__ = "Copyright 2021+, LPP"
__license__ = "Creative Commons 4.0 By-Nc-Sa"
__maintainer__ = "Roland Trouville"
__email__ = "roland.trouville@sorbonne-nouvelle.fr"
__status__ = "Production"

import wave

from PyQt5 import QtWidgets
from PyQt5.QtCore import Qt, pyqtSignal, QPoint
from PyQt5.QtGui import QFont, QIcon, QPainter, QPixmap, QMouseEvent, QCloseEvent
from PyQt5.QtWidgets import QComboBox, QGroupBox, QLabel, QLineEdit, QProgressBar, QPushButton, QScrollArea, QTextEdit, QMessageBox
import numpy as np
import pyqtgraph as pg
import math

from tools.data_manager import DataManager
from tools.file_manager import FileManager
from tools.video_manager import VideoManager


class QLabelDrawable(QLabel):
	clicked=pyqtSignal(QMouseEvent)

	def mousePressEvent(self, ev):
		self.clicked.emit(ev)

class MyPlotWidget(pg.PlotWidget):

	def __init__(self, *args, **kwargs):
		super().__init__(*args, **kwargs)
		self.scene().sigMouseClicked.connect(self.mouse_clicked)

	def mouse_clicked(self, mouseClickEvent):
		x = max(0,mouseClickEvent.pos()[0])
		tc = x / MainWindow.PIXEL_PER_SECOND
		# print(mouseClickEvent.pos()[0], tc)
		MainWindow.go_to_tc(tc)


class MainWindow(QtWidgets.QMainWindow):
	PIXEL_PER_SECOND = 200
	WAVEPLOT_X_PADDING = 0

	detail_group = None
	annotation_group = None
	save_group = None
	load_group = None

	wave_group = None
	wave_scroll = None
	wave_visu = None
	wave_plot = None
	wave_markerline = None
	wave_markerline_next = None
	wave_markerline_prev = None
	wave_x = None
	wave_y = None

	frame_group = None
	frame_group_prev = None
	frame_group_next = None
	frame_img = None
	frame_img_prev = None
	frame_img_next = None

	frame_grid = None
	frame_lines = None

	btn_gotoframe = None
	btn_gototc = None
	txt_frame = None
	txt_tc = None

	input_selector = None
	loader_progressbar = None

	annotation_textedit = {}
	annotation_labels = {}
	annotation_buttons = {}
	annotation_delete = {}
	annotation_values = {}

	export_start = None
	export_end = None

	logs_display = None

	vm = None
	current_frame_id = -1
	current_basepath = None
	current_line_drawing = None
	first_drawing_point = None



	def __init__(self, *args, **kwargs):
		super(MainWindow, self).__init__(*args, **kwargs)

		self.setGeometry(10, 30, 1351, 711)
		self.setFixedSize(1351,711)

		MainWindow.detail_group = QGroupBox(self)
		MainWindow.detail_group.move(0, 0)
		MainWindow.detail_group.resize(900, 711)

		# 3 frames display
		MainWindow.frame_group = QGroupBox(MainWindow.detail_group)
		MainWindow.frame_group.move(230, 0)
		MainWindow.frame_group.resize(660, 510)

		MainWindow.frame_group_prev = QGroupBox(MainWindow.detail_group)
		MainWindow.frame_group_prev.move(5, 0)
		MainWindow.frame_group_prev.resize(220, 180)

		MainWindow.frame_group_next = QGroupBox(MainWindow.detail_group)
		MainWindow.frame_group_next.move(5, 330)
		MainWindow.frame_group_next.resize(220, 180)

		MainWindow.frame_img = QLabel(MainWindow.frame_group)
		MainWindow.frame_img.move(10, 20)
		MainWindow.frame_img.resize(640, 480)
		MainWindow.frame_img.setAlignment(Qt.AlignTop)


		MainWindow.frame_img_prev = QLabel(MainWindow.frame_group_prev)
		MainWindow.frame_img_prev.move(10, 20)
		MainWindow.frame_img_prev.resize(200, 150)
		MainWindow.frame_img_prev.setAlignment(Qt.AlignTop)
		tmp = QLabel("T-1", MainWindow.frame_group_prev)
		tmp.move(175, 150)
		tmp.setFont(QFont("Arial", 20, QFont.Bold))
		tmp.setAutoFillBackground(True)
		tmp.setStyleSheet("background-color: rgba(255, 255, 0, 255)")
		tmp.raise_()

		MainWindow.frame_img_next = QLabel(MainWindow.frame_group_next)
		MainWindow.frame_img_next.move(10, 20)
		MainWindow.frame_img_next.resize(200, 150)
		MainWindow.frame_img_next.setAlignment(Qt.AlignTop)
		tmp = QLabel("T+1", MainWindow.frame_group_next)
		tmp.move(175, 150)
		tmp.setFont(QFont("Arial", 20, QFont.Bold))
		tmp.setAutoFillBackground(True)
		tmp.setStyleSheet("background-color: rgba(0, 255, 255, 255)")
		tmp.raise_()

		MainWindow.frame_grid = QLabel(MainWindow.frame_group)
		MainWindow.frame_grid.move(MainWindow.frame_img.pos())
		MainWindow.frame_grid.resize(MainWindow.frame_img.size())
		MainWindow.frame_grid.raise_()

		new_pix = QPixmap(MainWindow.frame_img.size())
		new_pix.fill(Qt.transparent)
		painter = QPainter(new_pix)
		painter.setOpacity(1)
		gridsize = 25
		for x in range(0,new_pix.width(),gridsize):
			painter.drawLine(x,0,x,new_pix.height())
		for y in range(0,new_pix.height(),gridsize):
			painter.drawLine(0,y,new_pix.width(),y)

		painter.end()
		MainWindow.frame_grid.setPixmap(new_pix)

		MainWindow.frame_lines = QLabelDrawable(MainWindow.frame_group)
		MainWindow.frame_lines.move(MainWindow.frame_img.pos())
		MainWindow.frame_lines.resize(MainWindow.frame_img.size())
		MainWindow.frame_lines.raise_()
		MainWindow.frame_lines.setVisible(True)
		MainWindow.frame_lines.clicked.connect(self.click_on_drawing)

	# Frame navigation

		tmp = QLabel("Frame:",MainWindow.detail_group)
		tmp.move(5,190)
		tmp.resize(40, 25)
		tmp.setAlignment(Qt.AlignCenter)

		MainWindow.txt_frame = QLineEdit("",MainWindow.detail_group)
		MainWindow.txt_frame.move(50,190)
		MainWindow.txt_frame.resize(100,25)
		MainWindow.btn_gotoframe = QPushButton("Go",MainWindow.detail_group)
		MainWindow.btn_gotoframe.move(160,190)
		MainWindow.btn_gotoframe.resize(50, 25)
		MainWindow.btn_gotoframe.clicked.connect(self.act_gotoframe)

		tmp = QLabel("Time:", MainWindow.detail_group)
		tmp.move(5, 220)
		tmp.resize(40, 25)
		tmp.setAlignment(Qt.AlignCenter)

		MainWindow.txt_tc = QLineEdit("", MainWindow.detail_group)
		MainWindow.txt_tc.move(50, 220)
		MainWindow.txt_tc.resize(100, 25)
		MainWindow.btn_gototc = QPushButton("Go", MainWindow.detail_group)
		MainWindow.btn_gototc.move(160, 220)
		MainWindow.btn_gototc.resize(50, 25)
		MainWindow.btn_gototc.clicked.connect(self.act_gototc)

		tmp = QPushButton("<<", MainWindow.detail_group)
		tmp.move(5, 250)
		tmp.resize(20, 25)
		tmp.clicked.connect(self.act_firstframe)
		tmp = QPushButton("< Previous", MainWindow.detail_group)
		tmp.move(30, 250)
		tmp.resize(70, 25)
		tmp.setAutoRepeat(True)
		tmp.clicked.connect(self.act_previousframe)
		tmp = QPushButton("Next >", MainWindow.detail_group)
		tmp.move(105, 250)
		tmp.resize(70, 25)
		tmp.setAutoRepeat(True)
		tmp.clicked.connect(self.act_nextframe)
		tmp = QPushButton(">>", MainWindow.detail_group)
		tmp.move(180, 250)
		tmp.resize(20, 25)
		tmp.clicked.connect(self.act_lastframe)

		tmp = QPushButton("Grid", MainWindow.detail_group)
		tmp.move(180, 310)
		tmp.resize(50, 25)
		tmp.clicked.connect(self.act_togglegrid)




		# The wav signal representation

		MainWindow.wave_group = QGroupBox("",self)
		MainWindow.wave_group.move(0, 520)
		MainWindow.wave_group.resize(900, 190)

		MainWindow.wave_scroll = QScrollArea(MainWindow.wave_group)
		MainWindow.wave_visu = MyPlotWidget(MainWindow.wave_scroll)
		MainWindow.wave_scroll.setWidget(MainWindow.wave_visu)
		MainWindow.wave_scroll.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOff)

		MainWindow.wave_scroll.move(0, 0)
		MainWindow.wave_scroll.resize(900, 190)
		MainWindow.wave_visu.resize(10000, 170)
		MainWindow.wave_visu.setBackground('k')
		MainWindow.wave_visu.setMouseEnabled(x=False, y=False)

		MainWindow.wave_markerline = pg.InfiniteLine(pos=0, angle=90, pen='r', movable=True)
		MainWindow.wave_visu.addItem(MainWindow.wave_markerline)
		MainWindow.wave_markerline.sigPositionChangeFinished.connect(self.line_changed)

		MainWindow.wave_markerline_prev = pg.InfiniteLine(pos=0, angle=90, pen='y', movable=False)
		MainWindow.wave_visu.addItem(MainWindow.wave_markerline_prev)

		MainWindow.wave_markerline_next = pg.InfiniteLine(pos=0, angle=90, pen='c', movable=False)
		MainWindow.wave_visu.addItem(MainWindow.wave_markerline_next)

		MainWindow.wave_visu.setAutoPan(x=True, y=False)

		# File loader
		gtmp = QGroupBox("Files", self)
		gtmp.move(905,5)
		gtmp.resize(441,80)

		MainWindow.input_selector = QComboBox(gtmp)
		MainWindow.input_selector.move(5,15)
		MainWindow.input_selector.resize(300,25)
		inputs = FileManager.list_all_inputs()
		for i in inputs:
			MainWindow.input_selector.addItem(i)
		tmp = QPushButton("Load", gtmp)
		tmp.move(310, 15)
		tmp.resize(50, 25)
		tmp.clicked.connect(self.act_loadfile)
		MainWindow.loader_progressbar = QProgressBar(gtmp)
		MainWindow.loader_progressbar.move(5, 50)
		MainWindow.loader_progressbar.resize(431, 25)

		MainWindow.load_group = gtmp

		# Annotations
		MainWindow.annotation_group = QGroupBox("Annotations",self)
		MainWindow.annotation_group.move(905,90)
		MainWindow.annotation_group.resize(441,426)
		MainWindow.annotation_group.setEnabled(False)



		#Lines -2 and -1 (special because no drawing possible)
		tmp = QLabelDrawable("", MainWindow.annotation_group)
		tmp.move(5,10)
		tmp.resize(205,20)
		tmp.clicked.connect(self.click_on_label)
		MainWindow.annotation_labels[-2] = tmp
		tmp = QTextEdit(MainWindow.annotation_group)
		tmp.move(5,30)
		tmp.resize(205, 30)
		tmp.setAcceptRichText(False)
		MainWindow.annotation_textedit[-2] = tmp
		tmp = QLabelDrawable("", MainWindow.annotation_group)
		tmp.move(215, 10)
		tmp.resize(205, 20)
		tmp.clicked.connect(self.click_on_label)
		MainWindow.annotation_labels[-1] = tmp
		tmp = QTextEdit(MainWindow.annotation_group)
		tmp.move(215, 30)
		tmp.resize(205, 30)
		tmp.setAcceptRichText(False)
		MainWindow.annotation_textedit[-1] = tmp



		#Lines 1 to DataManager.NB_LINES_MAX
		for i in range(DataManager.NB_LINES_MAX):
			tmp = QLabelDrawable("", MainWindow.annotation_group)
			tmp.move(5, math.floor(65+(i*60)))
			tmp.resize(431, 20)
			tmp.clicked.connect(self.click_on_label)
			MainWindow.annotation_labels[i] = tmp
			tmp = QTextEdit(MainWindow.annotation_group)
			tmp.move(40, math.floor(87+(i*60)))
			tmp.resize(300, 30)
			tmp.setAcceptRichText(False)
			MainWindow.annotation_textedit[i] = tmp

			tmp = QPushButton("",MainWindow.annotation_group)
			tmp.setCheckable(True)
			tmp.move(7, math.floor(87+(i*60)))
			tmp.resize(30,30)
			tmp.clicked.connect(self.start_drawing)
			tmp.setIcon(QIcon( QPixmap("./res/pencil.jpg")))
			tmp.setIconSize(tmp.size())
			MainWindow.annotation_buttons[i] = tmp

			tmp = QPushButton("", MainWindow.annotation_group)
			tmp.setCheckable(False)
			tmp.move(402, math.floor(87 + (i * 60)))
			tmp.resize(30, 30)
			tmp.setIcon(QIcon(QPixmap("./res/trash.png")))
			tmp.setIconSize(tmp.size())
			tmp.clicked.connect(self.delete_line)
			tmp.setVisible(False)
			MainWindow.annotation_delete[i] = tmp

			tmp = QLabel("0", MainWindow.annotation_group)
			tmp.move(350, math.floor(87 + (i * 60)))
			tmp.resize(45,35)
			tmp.setAlignment(Qt.AlignVCenter)
			tmp.setVisible(False)
			MainWindow.annotation_values[i] = tmp

		MainWindow.save_group = QGroupBox("Save and Export", self)
		MainWindow.save_group.move(905, 516)
		MainWindow.save_group.resize(441, 45)
		MainWindow.save_group.setEnabled(False)
		# MainWindow.save_group.setVisible(False)

		tmp = QPushButton("Save", MainWindow.save_group)
		tmp.move(5, 15)
		tmp.resize(80, 25)
		tmp.clicked.connect(MainWindow.save_current_frame_data)
		tmp.raise_()

		tmp = QLabel("Frame",MainWindow.save_group)
		tmp.move(200,15)
		tmp.resize(30,25)
		tmp.setAlignment(Qt.AlignVCenter)
		MainWindow.export_start = QLineEdit("",MainWindow.save_group)
		MainWindow.export_start.move(232,15)
		MainWindow.export_start.resize(50,25)
		tmp = QLabel("to", MainWindow.save_group)
		tmp.move(285, 15)
		tmp.resize(15, 25)
		tmp.setAlignment(Qt.AlignVCenter)
		MainWindow.export_end = QLineEdit("", MainWindow.save_group)
		MainWindow.export_end.move(302, 15)
		MainWindow.export_end.resize(50, 25)

		tmp = QPushButton("Export", MainWindow.save_group)
		tmp.move(360, 15)
		tmp.resize(70, 25)
		tmp.clicked.connect(self.export_data)



		# Log display
		gtmp = QGroupBox("Logs", self)
		gtmp.move(905, 561)
		gtmp.resize(441, 150)

		MainWindow.logs_display = QTextEdit(gtmp)
		MainWindow.logs_display.move(5,15)
		MainWindow.logs_display.resize(431,130)
		MainWindow.logs_display.setReadOnly(True)





		self.setWindowTitle("LPP - Physio Movie Annotation - v1.1.0")
		self.setWindowIcon(QIcon('./res/logo.png'))
		self.show()
		MainWindow.detail_group.setEnabled(False)

		#Temporary to gain time
		# MainWindow.load_data("./input/test1/")


	def export_data(self):
		try:
			DataManager.save_frame_data()
			try:
				frame_start = int(MainWindow.export_start.text())
			except ValueError:
				frame_start = 0
				MainWindow.export_start.setText("0")
			try:
				frame_end = int(MainWindow.export_end.text())
			except ValueError:
				frame_end = MainWindow.vm.last_frame_id
				MainWindow.export_end.setText(str(frame_end))



			DataManager.log("App: Trying to export from %d to %d"%(frame_start,frame_end))
			MainWindow.load_group.setEnabled(False)
			MainWindow.detail_group.setEnabled(False)
			MainWindow.annotation_group.setEnabled(False)
			MainWindow.save_group.setEnabled(False)
			fname = DataManager.export_frame_data(MainWindow.vm,frame_start, frame_end, "\t")
			tmp = QMessageBox(self)
			tmp.setWindowTitle(self.windowTitle())
			tmp.setText("Export is finished")
			tmp.setInformativeText("Results are in file "+fname)
			tmp.setModal(True)
			tmp.open()


			DataManager.log("App: Export Finished")
		except Exception as e:
			DataManager.log("App: Error while exporting "+repr(e))
		finally:
			MainWindow.load_group.setEnabled(True)
			MainWindow.detail_group.setEnabled(True)
			MainWindow.annotation_group.setEnabled(True)
			MainWindow.save_group.setEnabled(True)


	def start_drawing(self):
		sender_button = self.sender()
		if sender_button.isChecked():
			MainWindow.first_drawing_point = None
			for i in range(DataManager.NB_LINES_MAX):
				if sender_button == MainWindow.annotation_buttons[i]:
					MainWindow.current_line_drawing = i
				else:
					MainWindow.annotation_buttons[i].setChecked(False)
		else:
			MainWindow.current_line_drawing = None

	def delete_line(self):
		sender_button = self.sender()
		if MainWindow.current_frame_id is not None:
			MainWindow.first_drawing_point = None
			MainWindow.current_line_drawing = None
			for i in range(DataManager.NB_LINES_MAX):
				MainWindow.annotation_buttons[i].setChecked(False)
				if sender_button == MainWindow.annotation_delete[i]:
					DataManager.remove_line(MainWindow.current_frame_id, i)
					MainWindow.refresh_lines()

	@staticmethod
	def add_point(point:tuple):
		if MainWindow.current_line_drawing is not None and MainWindow.current_frame_id is not None:
			if MainWindow.first_drawing_point is None:
				MainWindow.first_drawing_point = point
			else:
				DataManager.add_line(MainWindow.current_frame_id, MainWindow.current_line_drawing, MainWindow.first_drawing_point, point)
				MainWindow.refresh_lines()
				MainWindow.annotation_buttons[MainWindow.current_line_drawing].setChecked(False)
				MainWindow.current_line_drawing = None
				MainWindow.first_drawing_point = None

	@staticmethod
	def refresh_lines():
		if MainWindow.current_frame_id is not None:
			MainWindow.frame_lines.setPixmap(
				DataManager.make_lines_pixmap(MainWindow.current_frame_id, MainWindow.frame_lines.size())
			)
			for i in range(DataManager.NB_LINES_MAX):
				val =DataManager.get_line_length(MainWindow.current_frame_id,i)
				if val != 0:
					MainWindow.annotation_values[i].setText("%0.1f"%val)
					MainWindow.annotation_values[i].setVisible(True)
					MainWindow.annotation_delete[i].setVisible(True)
				else:
					MainWindow.annotation_values[i].setVisible(False)
					MainWindow.annotation_delete[i].setVisible(False)


	def click_on_drawing(self, event:QMouseEvent):
		# print(event.pos())
		if event.button() == Qt.LeftButton:
			MainWindow.add_point( (event.pos().x(), event.pos().y()))
		else:
			MainWindow.current_line_drawing = None
			MainWindow.first_drawing_point = None
			for i in range(DataManager.NB_LINES_MAX):
				MainWindow.annotation_buttons[i].setChecked(False)

	def click_on_label(self, event:QMouseEvent):
		if event.button() == Qt.RightButton:
			sender_button = self.sender()
			for i in range(-2,DataManager.NB_LINES_MAX):
				if sender_button == MainWindow.annotation_labels[i]:
					name, ok = QtWidgets.QInputDialog.getText(
						self, 'Edit Label', 'Enter new label:', text=DataManager.get_line_label(i))
					if ok:
						DataManager.set_line_label(i, name)
						MainWindow.annotation_labels[i].setText(name)


	def act_gotoframe(self):
		inp = self.txt_frame.text()
		try:
			val = int(inp)
		except ValueError:
			val = 0
		MainWindow.display_frame(val)

	def act_nextframe(self):
		MainWindow.display_frame(self.current_frame_id+1)

	def act_previousframe(self):
		MainWindow.display_frame(self.current_frame_id-1)

	def act_firstframe(self):
		MainWindow.display_frame(0)

	def act_lastframe(self):
		MainWindow.display_frame(MainWindow.vm.last_frame_id)

	def act_gototc(self):
		inp = self.txt_tc.text()
		try:
			val = max(0,float(inp))
		except ValueError:
			val = 0
		frameid = MainWindow.vm.get_frame_id_at_tc(val)
		MainWindow.display_frame(frameid)

	def act_togglegrid(self):
		MainWindow.frame_grid.setVisible(not MainWindow.frame_grid.isVisible())

	def act_loadfile(self):
		MainWindow.load_data("./input/"+MainWindow.input_selector.currentText()+"/")


	@staticmethod
	def display_frame(target_frame_id):
		if target_frame_id < 0:
			target_frame_id = MainWindow.vm.last_frame_id
		try:
			tc = MainWindow.vm.get_tc_for_frame(target_frame_id)
		except KeyError:
			tc = 0

		if tc == 0:
			target_frame_id = 0

		if target_frame_id != MainWindow.current_frame_id:
			if 0 <= MainWindow.current_frame_id <= MainWindow.vm.last_frame_id:
				for i in range(-2,DataManager.NB_LINES_MAX):
					c = MainWindow.annotation_textedit[i].toPlainText()
					DataManager.add_comment(MainWindow.current_frame_id,i,c)
					MainWindow.annotation_textedit[i].clear()
			MainWindow.current_frame_id = target_frame_id
		else:
			return

		MainWindow.txt_tc.setText("%0.4f"%tc)
		MainWindow.txt_frame.setText(str(target_frame_id))
		MainWindow.wave_markerline.setValue(tc)
		target_x = math.floor(tc * MainWindow.PIXEL_PER_SECOND)
		MainWindow.wave_scroll.ensureVisible(target_x, 0, xMargin=200)

		base_title = "Frame %d / timecode %.04f"
		tmp = MainWindow.vm.get_frame(target_frame_id)
		tmp = tmp.scaled(MainWindow.frame_img.size(), Qt.KeepAspectRatio, Qt.SmoothTransformation)
		MainWindow.frame_img.setPixmap(tmp)
		MainWindow.refresh_lines()

		MainWindow.frame_group.setTitle(base_title % (target_frame_id, MainWindow.vm.get_tc_for_frame(target_frame_id)))
		if target_frame_id > 0:
			MainWindow.wave_markerline_prev.setValue(MainWindow.vm.get_tc_for_frame(target_frame_id - 1))
			tmp = MainWindow.vm.get_frame(target_frame_id - 1)
			tmp = tmp.scaled(MainWindow.frame_img_prev.size(), Qt.KeepAspectRatio, Qt.SmoothTransformation)
			MainWindow.frame_img_prev.setPixmap(tmp)
			MainWindow.frame_group_prev.setTitle(
				base_title % (target_frame_id - 1, MainWindow.vm.get_tc_for_frame(target_frame_id - 1)))
		else:
			MainWindow.wave_markerline_prev.setValue(tc)
			MainWindow.frame_img_prev.setPixmap(QPixmap())
			MainWindow.frame_group_prev.setTitle("")
		if target_frame_id < MainWindow.vm.last_frame_id - 1:
			MainWindow.wave_markerline_next.setValue(MainWindow.vm.get_tc_for_frame(target_frame_id + 1))
			tmp = MainWindow.vm.get_frame(target_frame_id + 1)
			tmp = tmp.scaled(MainWindow.frame_img_next.size(), Qt.KeepAspectRatio, Qt.SmoothTransformation)
			MainWindow.frame_img_next.setPixmap(tmp)
			MainWindow.frame_group_next.setTitle(
				base_title % (target_frame_id + 1, MainWindow.vm.get_tc_for_frame(target_frame_id + 1)))
		else:
			MainWindow.wave_markerline_next.setValue(tc)
			MainWindow.frame_img_next.setPixmap(QPixmap())
			MainWindow.frame_group_next.setTitle("")

		for i in range(-2,DataManager.NB_LINES_MAX):
			if i in MainWindow.annotation_textedit:
				MainWindow.annotation_textedit[i].append(DataManager.get_comment(target_frame_id,i))

		for i in range(DataManager.NB_LINES_MAX):
			MainWindow.annotation_buttons[i].setChecked(False)
		MainWindow.current_line_drawing = None
		MainWindow.first_drawing_point = None

	@staticmethod
	def line_changed():
		val = MainWindow.wave_markerline.value()
		MainWindow.go_to_tc(val)

	@staticmethod
	def go_to_tc(tc:float):
		if MainWindow.wave_x is None:
			return
		if tc < MainWindow.wave_x[0]:
			tc = MainWindow.wave_x[0]
		if tc > MainWindow.wave_x[-1]:
			tc = MainWindow.wave_x[-1]
		target_frame_id = MainWindow.vm.get_frame_id_at_tc(tc)
		# print("going to ",target_frame_id,tc)
		MainWindow.display_frame(target_frame_id)

	@staticmethod
	def load_data(path: str):
		#if we have current_basepath, we need to save all the annotations before changing of data
		if MainWindow.current_basepath is not None:
			DataManager.log("Annotations: Saving %s annotations before loading new data"%MainWindow.current_basepath)
			MainWindow.save_current_frame_data()
		MainWindow.current_basepath = path

		#Reset values for clean loading
		MainWindow.frame_img.setPixmap(QPixmap())
		MainWindow.frame_img_prev.setPixmap(QPixmap())
		MainWindow.frame_img_next.setPixmap(QPixmap())
		MainWindow.current_frame_id = -1



		MainWindow.loader_progressbar.setMaximum(4)
		MainWindow.loader_progressbar.setValue(0)
		DataManager.log("App: Loading input directory %s"%path)
		files = FileManager.check_input_dir(path)
		MainWindow.loader_progressbar.setValue(1)
		MainWindow.vm = VideoManager(files[0], path, MainWindow.loader_progressbar)

		MainWindow.loader_progressbar.setMaximum(4)
		MainWindow.loader_progressbar.setValue(2)
		if files[1] is not None:
			spf = wave.open(files[1], "r")

			# Extract Raw Audio from Wav File
			signal = spf.readframes(-1)
			nch = spf.getnchannels()
			framerate = spf.getframerate()
			nb_frames = spf.getnframes()
			DataManager.log("Wav File: %d channels, %d framerate, %d frames (%0.4f seconds)"%(nch,framerate, nb_frames, nb_frames/framerate))
			data = np.fromstring(signal, dtype=np.dtype(np.int16)) #Compatibilite numpy > 1.2
			ch_data = data[0::nch]
			spf.close()

			# Generate timecodes based on wav sampling rate
			MainWindow.wave_x = np.arange(0, len(ch_data))
			MainWindow.wave_x = MainWindow.wave_x / framerate

			# Truncate data up to the maximum time we have with frames (cutoff in VideoManager)
			MainWindow.wave_x = MainWindow.wave_x[ (MainWindow.wave_x <= MainWindow.vm.cutoff) ]
			if len(MainWindow.wave_x) < len(ch_data):
				DataManager.log("Wav File: truncating wave file to stop at last frame found in frames/ dir")
			MainWindow.wave_y = ch_data[0:len(MainWindow.wave_x)]
			DataManager.log("Wav File: last timecode of wav file in plot %0.4f seconds"%MainWindow.wave_x[-1])


			width =  math.floor(MainWindow.vm.cutoff * MainWindow.PIXEL_PER_SECOND)
			MainWindow.wave_visu.resize(width, 170)
			MainWindow.wave_visu.setYRange(-np.iinfo('int16').max, np.iinfo('int16').max)
			MainWindow.wave_visu.setXRange(MainWindow.wave_x[0], MainWindow.wave_x[-1], padding=0)
			MainWindow.wave_plot = MainWindow.wave_visu.plot(MainWindow.wave_x, MainWindow.wave_y)
			DataManager.log("Wav File: Downsampling to improve display speed")
			MainWindow.wave_plot.setDownsampling(auto=True, method='peak')
			MainWindow.wave_visu.hideAxis('left')
			MainWindow.wave_visu.getAxis('bottom').setTickSpacing(1, 1)
			MainWindow.wave_group.setVisible(True)
		else:
			MainWindow.wave_group.setVisible(False)

		MainWindow.detail_group.setEnabled(True)
		MainWindow.annotation_group.setEnabled(True)
		MainWindow.save_group.setEnabled(True)

		MainWindow.export_start.setText("0")
		MainWindow.export_end.setText(str(MainWindow.vm.last_frame_id))

		MainWindow.loader_progressbar.setValue(3)

		# DataManager.add_line(0,1,(22,44),(115,38))
		# DataManager.save_frame_data(MainWindow.current_basepath)

		DataManager.load_frame_data(path)
		for i in range(-2,DataManager.NB_LINES_MAX):
			if i in MainWindow.annotation_labels:
				MainWindow.annotation_labels[i].setText(DataManager.get_line_label(i))
				MainWindow.annotation_labels[i].setAlignment(Qt.AlignCenter)
				color = DataManager.get_line_color(i)
				if color is not None:
					MainWindow.annotation_labels[i].setAutoFillBackground(True)
					MainWindow.annotation_labels[i].setStyleSheet(
						"background-color: rgba("+DataManager.rgba_from_QColor(color)+")"
					)
		MainWindow.loader_progressbar.setValue(4)
		MainWindow.display_frame(0)

	@staticmethod
	def save_current_frame_data():
		try:
			DataManager.log("App: Manually saving annotations")
			DataManager.save_frame_data()
		except Exception as e:
			DataManager.log(e)

	def closeEvent(self, a0: QCloseEvent) -> None:
		if DataManager.current_basepath is not None:
			print("Saving annotations before closing app")
			DataManager.save_frame_data()
