#encoding utf-8

__author__ = "Roland Trouville"
__copyright__ = "Copyright 2021+, LPP"
__license__ = "Creative Commons 4.0 By-Nc-Sa"
__maintainer__ = "Roland Trouville"
__email__ = "roland.trouville@sorbonne-nouvelle.fr"
__status__ = "Production"

import os, datetime
from PyQt5.QtGui import QImage, QPixmap, QColor, QPainter, QPen
from PyQt5.QtCore import QSize, Qt, QPoint
import yaml
import time, math

class DataManager(object):

	AUTOSAVE_SECONDS_DELAY = 300 #5 minutes
	NB_LINES_MAX = 6

	current_basepath = None
	last_save = None

	# Data arrays
	frame_lines = None
	frame_comments = None
	line_colors = None
	line_labels = None

	qcolors_list = None



	@staticmethod
	def rgba_from_QColor(c:QColor, alpha:int=255)->str:
		if c is not None:
			return "{r}, {g}, {b}, {a}".format(r=c.red(),g=c.green(),b=c.blue(),a=alpha)
		return None

	@staticmethod
	def log(msg:str):
		from ui import MainWindow
		msg = datetime.datetime.today().strftime("[%H:%M] ")+msg
		print(msg)
		MainWindow.logs_display.append(msg)

	@staticmethod
	def load_frame_data(basepath:str):
		DataManager.frame_lines = {}
		DataManager.frame_comments = {}
		DataManager.line_colors = {}
		DataManager.line_labels = {}
		DataManager.current_basepath = basepath
		DataManager.last_save = time.time()
		if os.path.isfile(basepath+"data/lines.yaml"):
			f = open(basepath+"data/lines.yaml", "r", encoding="utf-8")
			DataManager.frame_lines = yaml.load(f, Loader=yaml.SafeLoader)
			f.close()
		if os.path.isfile(basepath + "data/comments.yaml"):
			f = open(basepath + "data/comments.yaml", "r", encoding="utf-8")
			DataManager.frame_comments = yaml.load(f, Loader=yaml.SafeLoader)
			f.close()
		if os.path.isfile(basepath + "data/colors.yaml"):
			f = open(basepath + "data/colors.yaml", "r", encoding="utf-8")
			DataManager.line_colors = yaml.load(f, Loader=yaml.FullLoader)
			f.close()
			# for k,v in tmp:
			# 	DataManager.line_colors[k] = (v[0],v[1],v[2])
		else:
			DataManager.line_colors = {
				0: (254,242,0), #Yellow
				1: (171,22,188), #Purple
				2: (1,163,230), #Blue
				3: (236,28,35), #Red
				4: (255,124,31), #Orange
				5: (34,177,78) #Green
			}
		if os.path.isfile(basepath + "data/labels.yaml"):
			f = open(basepath + "data/labels.yaml", "r", encoding="utf-8")
			DataManager.line_labels = yaml.load(f, Loader=yaml.FullLoader)
			f.close()
		else:
			DataManager.line_labels = {
				-2: "Wav", -1: "Frame", 0: "Line 1", 1: "Line 2", 2: "Line 3", 3: "Line 4",
				4: "Line 5", 5: "Line 6",
			}

	@staticmethod
	def save_frame_data():
		if DataManager.current_basepath is None:
			raise Exception("No path set for frame data, please load data before trying to save it")
		f = open(DataManager.current_basepath+"data/lines.yaml", "wt+", encoding="utf-8")
		yaml.dump(DataManager.frame_lines, f, Dumper=yaml.SafeDumper)
		f.close()
		f = open(DataManager.current_basepath + "data/comments.yaml", "wt+", encoding="utf-8")
		yaml.dump(DataManager.frame_comments, f, Dumper=yaml.SafeDumper)
		f.close()
		f = open(DataManager.current_basepath + "data/colors.yaml", "wt+", encoding="utf-8")
		yaml.dump(DataManager.line_colors, f, Dumper=yaml.Dumper)
		f.close()
		f = open(DataManager.current_basepath + "data/labels.yaml", "wt+", encoding="utf-8")
		yaml.dump(DataManager.line_labels, f, Dumper=yaml.Dumper)
		f.close()
		DataManager.last_save = time.time()

	@staticmethod
	def export_one_frame_data(frame_id: int, timecode:float, separator = "\t")->str:
		ret = str(frame_id)+separator+("%0.4f"%timecode)+separator+DataManager.get_comment(frame_id,-2)
		ret = ret +separator+DataManager.get_comment(frame_id,-1)
		for i in range(DataManager.NB_LINES_MAX):
			ret = ret+separator+("%0.1f"%DataManager.get_line_length(frame_id,i))+separator+DataManager.get_comment(frame_id,i)

		return ret

	@staticmethod
	def export_frame_data(vm,frame_start:int, frame_end: int, separator = "\t"):
		tstamp = datetime.datetime.now().isoformat(sep="_",timespec="minutes").replace(":","")
		fname = DataManager.current_basepath + "export/" + tstamp +".csv"
		file = open(fname, "w", encoding="utf-8")
		headers = "Frame number"+separator+"Timecode"+separator+"Frame"+separator+"Wav"
		for i in range(DataManager.NB_LINES_MAX):
			headers = headers + separator+DataManager.get_line_label(i) +separator+ DataManager.get_line_label(i)+"_com"
		file.write(headers+"\n")
		for i in range(frame_start,frame_end+1):
			tc = vm.get_tc_for_frame(i)
			file.write(DataManager.export_one_frame_data(i,tc,separator)+"\n")
		file.close()
		return fname


	@staticmethod
	def remove_line(frameid: int, linenumber: int):
		if frameid in DataManager.frame_lines:
			DataManager.frame_lines[frameid].pop(linenumber,None)

	@staticmethod
	def add_line(frameid: int, linenumber:int, firstpoint:tuple, secondpoint:tuple):
		if DataManager.frame_lines is None:
			DataManager.frame_lines = {}
		if frameid not in DataManager.frame_lines:
			DataManager.frame_lines[frameid] = {}
		DataManager.frame_lines[frameid][linenumber] = {
				"x1":firstpoint[0],
				"y1": firstpoint[1],
				"x2": secondpoint[0],
				"y2": secondpoint[1],
		}
		DataManager.check_autosave()

	@staticmethod
	def get_line(frameid:int, linenumber:int)->tuple:
		try:
			p = DataManager.frame_lines[frameid][linenumber]
			return (QPoint(p["x1"],p["y1"]),QPoint(p["x2"],p["y2"]))
		except KeyError:
			return None

	@staticmethod
	def get_line_length(frameid:int, linenumber:int)->float:
		l = DataManager.get_line(frameid,linenumber)
		if l is None:
			return 0
		else:
			return math.sqrt(((l[0].x() - l[1].x()) ** 2) + ( (l[0].y() - l[1].y()) ** 2 ))

	@staticmethod
	def get_line_label(linenumber:int)->str:
		try:
			return DataManager.line_labels[linenumber]
		except KeyError:
			return ""

	@staticmethod
	def set_line_label(linenumber: int, label: str):
		DataManager.line_labels[linenumber] = label
		DataManager.check_autosave()

	@staticmethod
	def get_comment(frameid:int, linenumber:int)->str:
		try:
			return DataManager.frame_comments[frameid][linenumber]
		except KeyError:
			return ""

	@staticmethod
	def add_comment(frameid: int, linenumber: int, comment:str):
		if comment is None or comment == "":
			if frameid in DataManager.frame_comments:
				DataManager.frame_comments[frameid].pop(linenumber, None)
		else:
			if DataManager.frame_comments is None:
				DataManager.frame_comments = {}
			if frameid not in DataManager.frame_comments:
				DataManager.frame_comments[frameid] = {}
			DataManager.frame_comments[frameid][linenumber] = comment
		DataManager.check_autosave()

	@staticmethod
	def check_autosave():
		ts = time.time()
		if ts > DataManager.last_save + DataManager.AUTOSAVE_SECONDS_DELAY:
			DataManager.log("Annotations: Auto-saving data for %s" % DataManager.current_basepath)
			DataManager.save_frame_data()
			DataManager.last_save = ts

	@staticmethod
	def get_line_color(linenumber:int)->QColor:
		try:
			t = DataManager.line_colors[linenumber]
		except KeyError:
			return None
		if DataManager.qcolors_list is None:
			DataManager.qcolors_list = {}
		if t not in DataManager.qcolors_list:
			DataManager.qcolors_list[t] = QColor(t[0],t[1],t[2])
		return DataManager.qcolors_list[t]

	@staticmethod
	def make_lines_pixmap(frame_id:int, size:QSize)->QPixmap:
		new_pix = QPixmap(size)
		new_pix.fill(Qt.transparent)
		painter = QPainter(new_pix)
		painter.setOpacity(1)
		for i in range(DataManager.NB_LINES_MAX):
			l = DataManager.get_line(frame_id,i)
			if l is not None:
				# print("hop",DataManager.line_colors[i],l[0],l[1])
				ptemp = QPen(DataManager.get_line_color(i))
				ptemp.setWidth(3)
				painter.setPen(ptemp)
				painter.drawLine(l[0],l[1])
		painter.end()
		return new_pix
